/*
 * last modified---
 * 	07-12-22 improve error output labels
 * 	06-29-22 add convenience method getKeyForHash()
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-25-22 new
 *
 * purpose---
 * 	encapsulate reply messages from Auditors to MVOs related to keyring requests
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.ArrayList;
import java.util.Map;
import java.text.NumberFormat;
import java.io.IOException;


/**
 * This class holds the data necessary to build JSON objects related to keys
 * which are sent to MVOs by Auditors as replies.  This class knows how to emit
 * itself as JSON. Because MVOs need to parse these replies, it also knows how
 * to build itself from a JSON string.
 */
public final class AuditorKeyBlock implements JSON.Generator {
	// BEGIN data members
	/**
	 * constant for success status return
	 */
	public final String			M_Success = "success";

	/**
	 * return status of the receipt request, either "success" or an error text
	 */
	private String				m_Status;

	/**
	 * helper class describing a key retrieved
	 */
	public class KeySpec {
		/**
		 * the receipt index (mapEntry001, mapEntry002, etc.)
		 */
		public String		m_Sequence;

		/**
		 * the index hash (echo of value supplied in the MVO's request)
		 */
		public String		m_Hash;

		/**
		 * the AES-256 key data, base64Url-encoded
		 */
		public String		m_KeyData;

		/**
		 * nullary constructor
		 */
		public KeySpec() {
			m_Sequence = m_Hash = m_KeyData = "";
		}
	}

	/**
	 * list of keys being returned
	 */
	public ArrayList<KeySpec>	m_Keys;

	/**
	 * helper class to describe Auditor signature on the receipt operation reply
	 */
	public final class AuditorSignature {
		/**
		 * the signer (the ID of the Auditor who signs)
		 */
		public String			m_Signer;

		/**
		 * the actual signature, in base64 format
		 */
		public String			m_Signature;

		/**
		 * nullary constructor
		 */
		public AuditorSignature() {
			m_Signer = m_Signature = "";
		}
	}

	/**
	 * the attached Auditor signature
	 */
	public AuditorSignature		m_Signature;

	/**
	 * logging object
	 */
	private Log					m_Log;

	/**
	 * error code status (must be 0 to indicate success)
	 */
	private int					m_ErrCode;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param logger the logging object
	 */
	public AuditorKeyBlock(Log logger) {
		m_Log = logger;
		m_Status = M_Success;
		m_Signature = new AuditorSignature();
		m_Keys = new ArrayList<KeySpec>();
	}

	// GET methods
	/**
	 * obtain the status message
	 * @return the return status, either "success" or an error message
	 */
	public String getStatus() { return m_Status; }

	/**
	 * obtain the list of key outputs for this reply block
	 * @return the list, empty if none
	 */
	public ArrayList<KeySpec> getKeys() { return m_Keys; }

	/**
	 * obtain the key spec entry for a given hash
	 * @param hash the hash to search for
	 * @return the key spec, or null if not found in the array
	 */
	public KeySpec getKeyForHash(String hash) {
		if (hash == null || hash.isEmpty()) {
			return null;
		}
		for (KeySpec kSpec : m_Keys) {
			if (kSpec.m_Hash.equals(hash)) {
				return kSpec;
			}
		}
		return null;
	}

	/**
	 * obtain the Auditor signature for this reply block
	 */
	public AuditorSignature getSignature() { return m_Signature; }

	/**
	 * obtain the error code from the last addJSON() call
	 * @return the result code from building output JSON
	 */
	public int getErrCode() { return m_ErrCode; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * config the status return value (either M_Success or an error value)
	 * @param stat the status to set
	 */
	public void setStatus(String stat) {
		if (stat != null && !stat.isEmpty()) {
			m_Status = new String(stat);
		}
		else {
			m_Log.error("AuditorKeyBlock.setStatus: missing status return "
						+ "value");
			m_Status = M_Success;
		}
	}

	/**
	 * add a key to the output list
	 * @param rct a receipt descriptor
	 */
	public void addKey(KeySpec key) {
		if (key != null) {
			m_Keys.add(key);
		}
	}

	/**
	 * configure the signature on the signed fields
	 * @param sig the signature of the Auditor generating this reply
	 */
	public void setSignature(AuditorSignature sig) {
		if (sig != null) {
			m_Signature = sig;
		}
		else {
			m_Log.error("AuditorKeyBlock.setSignature: missing Auditor "
						+ "signature");
		}
	}


	/**
	 * Build the JSON text that's actually signed.
	 * @return the JSON string for the element without delimiters, or null on
	 * error (m_ErrCode set to some appropriate value)
	 */
	public String buildSignedData() {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		final String lbl
			= this.getClass().getSimpleName() + ".buildSignedData: ";

		// add status
		if (m_Status.isEmpty()) {
			m_ErrCode = 1;
			m_Log.error(lbl + "missing status return value");
			return null;
		}
		out.append("\"status\":\"" + m_Status + "\",");

		int kIdx = 1;
		out.append("\"keys\":[");
		for (KeySpec kSpec : m_Keys) {
			kSpec.m_Sequence = "mapEntry" + nf.format(kIdx++);
			if (kSpec.m_Hash.isEmpty()) {
				m_ErrCode = 2;
				m_Log.error("Key " + kSpec.m_Sequence + " has missing hash");
				return null;
			}
			if (kSpec.m_KeyData.isEmpty()) {
				m_ErrCode = 3;
				m_Log.error("Key " + kSpec.m_Sequence
							+ " has missing key data");
				return null;
			}
			out.append("{\"" + kSpec.m_Sequence + "\":{");
			out.append("\"hash\":\"" + kSpec.m_Hash + "\",");
			out.append("\"key\":\"" + kSpec.m_KeyData + "\"");
			out.append("}}");
			// add comma except on last
			if (kIdx <= m_Keys.size()) {
				out.append(",");
			}
		}
		out.append("]");

		// NB: signature should be added to this
		return out.toString();
	}

	/**
	 * method to build the entire reply object from a JSON string
	 * @param repData the reply data (a JSON object)
	 * @return true on success
	 */
	public boolean buildFromString(String repData) {
		boolean ret = false;
		final String lbl
			= this.getClass().getSimpleName() + ".buildFromString: ";
		if (repData == null || repData.isEmpty()) {
			m_Log.error(lbl + "missing key reply data");
			return ret;
		}
		Object rep = null;
		try {
			rep = JSON.parse(repData);
		}
		catch (IllegalStateException ise) {
			m_Log.error(lbl + "client JSON did not parse: \"" + repData + "\"");
			return ret;
		}

		// parsed object should consist of a Map
		if (!(rep instanceof Map)) {
			m_Log.error(lbl + "client JSON was not a Map");
			return ret;
		}
		Map map = (Map) rep;
		ret = buildFromMap(map);
		return ret;
	}

	/**
	 * build the object from a Map, the result of a JSON parse
	 * @param block the mapping
	 * @return true on success
	 */
	public boolean buildFromMap(Map block) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (block == null || block.isEmpty()) {
			m_Log.error(lbl + "missing top Map");
			return false;
		}
		boolean ret = true;

		// required: the status
		Object stat = block.get("status");
		if (stat instanceof String) {
			String status = (String) stat;
			setStatus(status);
		}
		else {
			m_Log.error(lbl + "missing status return value");
			ret = false;
			m_ErrCode = 5;
		}

		// required: key list
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		Object keyList = block.get("keys");
		if (ret && keyList instanceof Object[]) {
			Object[] keys = (Object[]) keyList;
			for (int kkk = 0; kkk < keys.length; kkk++) {
				Object kList = keys[kkk];
				if (kList instanceof Map) {
					Map keySpec = (Map) kList;
					// get input index
					long idx = kkk + 1;
					String entry = "mapEntry" + nf.format(idx);
					Object spec = keySpec.get(entry);
					KeySpec ks = new KeySpec();
					if (spec instanceof Map) {
						Map kSpec = (Map) spec;
						ks.m_Sequence = new String(entry);

						// get the hash
						Object hashO = kSpec.get("hash");
						if (hashO instanceof String) {
							String hash = (String) hashO;
							ks.m_Hash = new String(hash);
						}
						else {
							ret = false;
							m_Log.error(lbl + "missing hash for "
										+ ks.m_Sequence);
						}

						// get the key data
						Object keyO = kSpec.get("key");
						if (keyO instanceof String) {
							String keyDat = (String) keyO;
							ks.m_KeyData = new String(keyDat);
							addKey(ks);
						}
						else {
							ret = false;
							m_Log.error(lbl + "missing key data for "
										+ ks.m_Sequence);
						}
					}
					else {
						ret = false;
						m_Log.error(lbl + "key data for " + entry
									+ " was not a Map");
					}
				}
				else {
					ret = false;
					m_Log.error(lbl + "key array item " + (kkk+1)
								+ " was not a Map");
				}
				if (!ret) break;
			}
			if (!ret) {
				m_ErrCode = 7;
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing key list");
			m_ErrCode = 6;
		}

		// get the signature
		Object sigBlock = block.get("signature");
		if (ret && sigBlock instanceof Map) {
			Map sigSpec = (Map) sigBlock;
			AuditorSignature sig = new AuditorSignature();

			// get Auditor signer ID
			Object signer = sigSpec.get("signer");
			if (signer instanceof String) {
				String audID = (String) signer;
				sig.m_Signer = new String(audID);
			}
			else {
				ret = false;
				m_Log.error(lbl + "key reply sig missing signer");
			}

			// get actual signature
			Object aSig = sigSpec.get("sig");
			if (aSig instanceof String) {
				String audSig = (String) aSig;
				sig.m_Signature = new String(audSig);
			}
			else {
				ret = false;
				m_Log.error(lbl + "key reply sig missing signature");
			}
			if (ret) {
				m_Signature = sig;
			}
			else {
				m_ErrCode = 9;
			}
		}
		else {
			m_Log.error(lbl + "missing Auditor signature");
			ret = false;
			m_ErrCode = 8;
		}

		return ret;
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		out.append("{");
		// first, put the signed data in
		String audSigned = buildSignedData();
		if (audSigned == null || m_ErrCode != 0) {
			return;
		}
		out.append(audSigned);

		// NB: caller is responsible for having added m_Signature before this
		if (m_Signature.m_Signature.isEmpty()) {
			m_Log.error("AuditorKeyBlock.addJSON(): missing signature");
			m_ErrCode = 4;
			return;
		}
		out.append(",\"signature\":{");
		out.append("\"signer\":\"" + m_Signature.m_Signer + "\",");
		out.append("\"sig\":\"" + m_Signature.m_Signature + "\"");
		out.append("}}");	// end Signature obj, outer obj
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("AuditorKeyBlock.addJSON(): exception appending", ioe);
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Keys != null) {
				m_Keys.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
